import fs from 'fs';
import path from 'path';
import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();
const DATA_DIR = path.join(process.cwd(), 'data');
const DB_FILE = path.join(DATA_DIR, 'projects.json');

async function main() {
    if (!fs.existsSync(DB_FILE)) {
        console.log('No projects.json found, skipping migration.');
        return;
    }

    const data = fs.readFileSync(DB_FILE, 'utf-8');
    const projects = JSON.parse(data);

    console.log(`Found ${projects.length} projects to migrate...`);

    for (const project of projects) {
        await prisma.project.upsert({
            where: { id: project.id },
            update: {
                templateId: project.templateId,
                name: project.name,
                slug: project.slug,
                data: JSON.stringify(project.data),
                createdAt: new Date(project.createdAt),
                views: project.views || 0,
            },
            create: {
                id: project.id,
                templateId: project.templateId,
                name: project.name,
                slug: project.slug,
                data: JSON.stringify(project.data),
                createdAt: new Date(project.createdAt),
                views: project.views || 0,
            },
        });
        console.log(`Migrated project: ${project.name} (${project.id})`);
    }

    console.log('Migration complete!');
}

main()
    .catch((e) => {
        console.error(e);
        process.exit(1);
    })
    .finally(async () => {
        await prisma.$disconnect();
    });
