#!/bin/bash
# Restore Script for Landing Page Builder
# Restores database and files from backup

# Configuration
BACKUP_DIR="$HOME/backups"
APP_DIR="$HOME/app"
DB_NAME="your_database_name"
DB_USER="your_database_user"
DB_PASS="your_database_password"

# Check if backup directory exists
if [ ! -d "$BACKUP_DIR" ]; then
    echo "❌ Backup directory not found: $BACKUP_DIR"
    exit 1
fi

echo "🔄 Starting restore process..."
echo ""

# List available backups
echo "📋 Available database backups:"
ls -lh "$BACKUP_DIR"/db_backup_*.sql.gz 2>/dev/null | tail -5
echo ""
echo "📋 Available application backups:"
ls -lh "$BACKUP_DIR"/app_backup_*.tar.gz 2>/dev/null | tail -5
echo ""

# Get latest backups by default
LATEST_DB=$(ls -t "$BACKUP_DIR"/db_backup_*.sql.gz 2>/dev/null | head -1)
LATEST_APP=$(ls -t "$BACKUP_DIR"/app_backup_*.tar.gz 2>/dev/null | head -1)

if [ -z "$LATEST_DB" ] || [ -z "$LATEST_APP" ]; then
    echo "❌ No backups found!"
    exit 1
fi

echo "🎯 Latest database backup: $(basename $LATEST_DB)"
echo "🎯 Latest app backup: $(basename $LATEST_APP)"
echo ""

# Confirmation
read -p "⚠️  This will overwrite existing data. Continue? (yes/no): " confirm
if [ "$confirm" != "yes" ]; then
    echo "❌ Restore cancelled"
    exit 0
fi

# 1. Stop the application
echo "🛑 Stopping application..."
pm2 stop all
sleep 2

# 2. Restore Database
echo "💾 Restoring database..."
gunzip -c "$LATEST_DB" | mysql -u "$DB_USER" -p"$DB_PASS" "$DB_NAME"

if [ $? -eq 0 ]; then
    echo "✅ Database restored successfully"
else
    echo "❌ Database restore failed!"
    exit 1
fi

# 3. Restore Application Files
echo "📦 Restoring application files..."
read -p "⚠️  Restore application files? This will overwrite existing files (yes/no): " confirm_app
if [ "$confirm_app" = "yes" ]; then
    # Backup current files first
    echo "📋 Creating safety backup of current files..."
    tar -czf "$APP_DIR/../app_before_restore_$(date +%Y%m%d_%H%M%S).tar.gz" -C "$APP_DIR" .
    
    # Extract backup
    tar -xzf "$LATEST_APP" -C "$APP_DIR"
    
    if [ $? -eq 0 ]; then
        echo "✅ Application files restored successfully"
    else
        echo "❌ Application files restore failed!"
        exit 1
    fi
fi

# 4. Restore environment variables
LATEST_ENV=$(ls -t "$BACKUP_DIR"/env_backup_* 2>/dev/null | head -1)
if [ -n "$LATEST_ENV" ]; then
    read -p "🔐 Restore .env file? (yes/no): " confirm_env
    if [ "$confirm_env" = "yes" ]; then
        cp "$LATEST_ENV" "$APP_DIR/.env"
        chmod 600 "$APP_DIR/.env"
        echo "✅ Environment variables restored"
    fi
fi

# 5. Reinstall dependencies if needed
if [ "$confirm_app" = "yes" ]; then
    echo "📦 Installing dependencies..."
    cd "$APP_DIR"
    npm install --production
    npx prisma generate
fi

# 6. Restart application
echo "🚀 Starting application..."
pm2 restart all

sleep 3
pm2 status

echo ""
echo "✅ Restore process completed!"
echo ""
echo "🔍 Check application status:"
echo "   - pm2 status"
echo "   - pm2 logs"
echo "   - curl http://localhost:3000/api/health"
echo ""

exit 0

