#!/bin/bash
# سكريبت الإعداد التلقائي الكامل لـ VPS
# يثبت كل حاجة من الصفر!

set -e  # إيقاف عند أي خطأ

echo "🚀 إعداد تلقائي كامل لـ VPS"
echo "================================"
echo ""

# ألوان
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
BLUE='\033[0;34m'
NC='\033[0m'

# التحقق من صلاحيات root
if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}❌ يجب تشغيل السكريبت بصلاحيات root${NC}"
    echo "استخدم: sudo ./setup-vps-auto.sh"
    exit 1
fi

echo -e "${BLUE}📝 معلومات السيرفر:${NC}"
echo "OS: $(cat /etc/os-release | grep PRETTY_NAME | cut -d'"' -f2)"
echo "IP: $(hostname -I | awk '{print $1}')"
echo ""

# 1. تحديث النظام
echo -e "${YELLOW}📦 الخطوة 1/10: تحديث النظام...${NC}"
apt-get update -qq
apt-get upgrade -y -qq
echo -e "${GREEN}✅ تم تحديث النظام${NC}"
echo ""

# 2. تثبيت Node.js
echo -e "${YELLOW}📦 الخطوة 2/10: تثبيت Node.js 20.x...${NC}"
if ! command -v node &> /dev/null; then
    curl -fsSL https://deb.nodesource.com/setup_20.x | bash -
    apt-get install -y nodejs
    echo -e "${GREEN}✅ تم تثبيت Node.js $(node -v)${NC}"
else
    echo -e "${GREEN}✅ Node.js موجود بالفعل: $(node -v)${NC}"
fi
echo ""

# 3. تثبيت MySQL
echo -e "${YELLOW}🗄️  الخطوة 3/10: تثبيت MySQL...${NC}"
if ! command -v mysql &> /dev/null; then
    export DEBIAN_FRONTEND=noninteractive
    apt-get install -y mysql-server
    systemctl start mysql
    systemctl enable mysql
    echo -e "${GREEN}✅ تم تثبيت MySQL${NC}"
else
    echo -e "${GREEN}✅ MySQL موجود بالفعل${NC}"
fi
echo ""

# 4. تثبيت PM2
echo -e "${YELLOW}📦 الخطوة 4/10: تثبيت PM2...${NC}"
if ! command -v pm2 &> /dev/null; then
    npm install -g pm2
    echo -e "${GREEN}✅ تم تثبيت PM2${NC}"
else
    echo -e "${GREEN}✅ PM2 موجود بالفعل${NC}"
fi
echo ""

# 5. تثبيت Nginx
echo -e "${YELLOW}🌐 الخطوة 5/10: تثبيت Nginx...${NC}"
if ! command -v nginx &> /dev/null; then
    apt-get install -y nginx
    systemctl start nginx
    systemctl enable nginx
    echo -e "${GREEN}✅ تم تثبيت Nginx${NC}"
else
    echo -e "${GREEN}✅ Nginx موجود بالفعل${NC}"
fi
echo ""

# 6. إنشاء مستخدم للتطبيق
echo -e "${YELLOW}👤 الخطوة 6/10: إنشاء مستخدم التطبيق...${NC}"
APP_USER="appuser"
if ! id "$APP_USER" &>/dev/null; then
    useradd -m -s /bin/bash $APP_USER
    echo -e "${GREEN}✅ تم إنشاء المستخدم: $APP_USER${NC}"
else
    echo -e "${GREEN}✅ المستخدم موجود: $APP_USER${NC}"
fi
echo ""

# 7. إنشاء قاعدة البيانات
echo -e "${YELLOW}🗄️  الخطوة 7/10: إعداد قاعدة البيانات...${NC}"
DB_NAME="landing_pages"
DB_USER="landing_user"
DB_PASS=$(openssl rand -base64 16)

mysql -e "CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;" 2>/dev/null || true
mysql -e "CREATE USER IF NOT EXISTS '$DB_USER'@'localhost' IDENTIFIED BY '$DB_PASS';" 2>/dev/null || true
mysql -e "GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USER'@'localhost';" 2>/dev/null || true
mysql -e "FLUSH PRIVILEGES;" 2>/dev/null || true

echo -e "${GREEN}✅ تم إعداد قاعدة البيانات${NC}"
echo ""

# حفظ بيانات قاعدة البيانات
DB_CREDENTIALS="/root/db-credentials.txt"
cat > $DB_CREDENTIALS << EOF
Database Name: $DB_NAME
Database User: $DB_USER
Database Password: $DB_PASS
Database URL: mysql://$DB_USER:$DB_PASS@localhost:3306/$DB_NAME
EOF
chmod 600 $DB_CREDENTIALS
echo -e "${BLUE}📝 تم حفظ بيانات قاعدة البيانات في: $DB_CREDENTIALS${NC}"
echo ""

# 8. إعداد مجلد التطبيق
echo -e "${YELLOW}📁 الخطوة 8/10: إعداد مجلد التطبيق...${NC}"
APP_DIR="/home/$APP_USER/app"
mkdir -p $APP_DIR
chown -R $APP_USER:$APP_USER $APP_DIR
echo -e "${GREEN}✅ مجلد التطبيق جاهز: $APP_DIR${NC}"
echo ""

# 9. نسخ الملفات (إذا كانت موجودة)
echo -e "${YELLOW}📦 الخطوة 9/10: نسخ ملفات التطبيق...${NC}"
CURRENT_DIR=$(pwd)
if [ -f "$CURRENT_DIR/package.json" ]; then
    echo "نسخ الملفات من $CURRENT_DIR إلى $APP_DIR..."
    cp -r $CURRENT_DIR/* $APP_DIR/ 2>/dev/null || true
    chown -R $APP_USER:$APP_USER $APP_DIR
    echo -e "${GREEN}✅ تم نسخ الملفات${NC}"
else
    echo -e "${YELLOW}⚠️  لم يتم العثور على ملفات التطبيق هنا${NC}"
    echo "ارفع الملفات إلى: $APP_DIR"
fi
echo ""

# 10. إنشاء ملف .env
echo -e "${YELLOW}🔐 الخطوة 10/10: إنشاء ملف .env...${NC}"
read -p "🌐 أدخل رابط الموقع (مثل: https://site.mdoante.com): " SITE_URL
NEXTAUTH_SECRET=$(openssl rand -base64 32)

cat > $APP_DIR/.env << EOF
DATABASE_URL="mysql://$DB_USER:$DB_PASS@localhost:3306/$DB_NAME"
NEXTAUTH_URL="$SITE_URL"
NEXTAUTH_SECRET="$NEXTAUTH_SECRET"
NODE_ENV="production"
PORT=3000
EOF

chmod 600 $APP_DIR/.env
chown $APP_USER:$APP_USER $APP_DIR/.env
echo -e "${GREEN}✅ تم إنشاء ملف .env${NC}"
echo ""

# إعداد التطبيق
if [ -f "$APP_DIR/package.json" ]; then
    echo -e "${YELLOW}📦 تثبيت مكتبات التطبيق...${NC}"
    cd $APP_DIR
    sudo -u $APP_USER npm install --production
    
    echo -e "${YELLOW}🔧 إعداد Prisma...${NC}"
    sudo -u $APP_USER npx prisma generate
    sudo -u $APP_USER npx prisma migrate deploy
    
    # إنشاء مجلد logs
    mkdir -p $APP_DIR/logs
    chown $APP_USER:$APP_USER $APP_DIR/logs
    
    echo -e "${GREEN}✅ تم إعداد التطبيق${NC}"
fi
echo ""

# إعداد Nginx
echo -e "${YELLOW}🌐 إعداد Nginx...${NC}"
DOMAIN=$(echo $SITE_URL | sed 's~http[s]*://~~g')
NGINX_CONF="/etc/nginx/sites-available/landing-pages"

cat > $NGINX_CONF << 'EOF'
server {
    listen 80;
    server_name DOMAIN_PLACEHOLDER;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}
EOF

sed -i "s/DOMAIN_PLACEHOLDER/$DOMAIN/g" $NGINX_CONF

ln -sf $NGINX_CONF /etc/nginx/sites-enabled/
rm -f /etc/nginx/sites-enabled/default
nginx -t && systemctl reload nginx

echo -e "${GREEN}✅ تم إعداد Nginx${NC}"
echo ""

# بدء التطبيق
if [ -f "$APP_DIR/ecosystem.config.js" ]; then
    echo -e "${YELLOW}🚀 بدء التطبيق...${NC}"
    cd $APP_DIR
    sudo -u $APP_USER pm2 start ecosystem.config.js
    sudo -u $APP_USER pm2 save
    
    # إعداد PM2 للبدء التلقائي
    env PATH=$PATH:/usr/bin pm2 startup systemd -u $APP_USER --hp /home/$APP_USER
    
    echo -e "${GREEN}✅ التطبيق يعمل الآن!${NC}"
fi
echo ""

# معلومات SSL (اختياري)
echo -e "${YELLOW}🔒 لتفعيل SSL (HTTPS):${NC}"
echo "apt-get install -y certbot python3-certbot-nginx"
echo "certbot --nginx -d $DOMAIN"
echo ""

# ملخص نهائي
echo "================================"
echo -e "${GREEN}🎉 الإعداد مكتمل!${NC}"
echo "================================"
echo ""
echo -e "${BLUE}📋 معلومات مهمة:${NC}"
echo ""
echo "🌐 رابط الموقع: $SITE_URL"
echo "📁 مجلد التطبيق: $APP_DIR"
echo "👤 مستخدم التطبيق: $APP_USER"
echo "🗄️  بيانات قاعدة البيانات: $DB_CREDENTIALS"
echo ""
echo -e "${BLUE}🛠️  أوامر مفيدة:${NC}"
echo ""
echo "# حالة التطبيق"
echo "sudo -u $APP_USER pm2 status"
echo ""
echo "# السجلات"
echo "sudo -u $APP_USER pm2 logs"
echo ""
echo "# إعادة التشغيل"
echo "sudo -u $APP_USER pm2 restart all"
echo ""
echo "# حالة Nginx"
echo "systemctl status nginx"
echo ""
echo "================================"
echo -e "${GREEN}✨ بالتوفيق! ✨${NC}"
echo "================================"

