// Health Check Endpoint
// Used to monitor application status

import { NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';

export async function GET() {
    try {
        const startTime = Date.now();
        
        // Check database connection
        await prisma.$queryRaw`SELECT 1`;
        
        const dbResponseTime = Date.now() - startTime;
        
        // Check memory usage
        const memUsage = process.memoryUsage();
        const memUsageMB = {
            rss: Math.round(memUsage.rss / 1024 / 1024),
            heapTotal: Math.round(memUsage.heapTotal / 1024 / 1024),
            heapUsed: Math.round(memUsage.heapUsed / 1024 / 1024),
            external: Math.round(memUsage.external / 1024 / 1024),
        };
        
        // Check uptime
        const uptime = process.uptime();
        const uptimeFormatted = {
            hours: Math.floor(uptime / 3600),
            minutes: Math.floor((uptime % 3600) / 60),
            seconds: Math.floor(uptime % 60),
        };
        
        return NextResponse.json({
            status: 'healthy',
            timestamp: new Date().toISOString(),
            uptime: uptimeFormatted,
            database: {
                connected: true,
                responseTime: `${dbResponseTime}ms`,
            },
            memory: memUsageMB,
            environment: process.env.NODE_ENV,
            version: process.env.npm_package_version || '0.1.0',
        });
    } catch (error) {
        console.error('Health check failed:', error);
        
        return NextResponse.json(
            {
                status: 'unhealthy',
                timestamp: new Date().toISOString(),
                error: 'Database connection failed',
            },
            { status: 503 }
        );
    }
}

// Disable caching for health checks
export const dynamic = 'force-dynamic';
export const revalidate = 0;

