import { auth } from "@/auth"
import { NextResponse } from "next/server"

export default auth((req) => {
    // Get response
    const response = NextResponse.next();
    
    // Add security headers
    response.headers.set('X-Content-Type-Options', 'nosniff');
    response.headers.set('X-Frame-Options', 'SAMEORIGIN');
    response.headers.set('X-XSS-Protection', '1; mode=block');
    response.headers.set('Referrer-Policy', 'strict-origin-when-cross-origin');
    
    // Add HSTS in production
    if (process.env.NODE_ENV === 'production') {
        response.headers.set(
            'Strict-Transport-Security',
            'max-age=31536000; includeSubDomains'
        );
    }
    
    // Log suspicious requests
    const userAgent = req.headers.get('user-agent') || '';
    const suspiciousPatterns = [
        /sqlmap/i,
        /nikto/i,
        /nmap/i,
        /masscan/i,
        /acunetix/i
    ];
    
    if (suspiciousPatterns.some(pattern => pattern.test(userAgent))) {
        console.warn('Suspicious request detected:', {
            ip: req.headers.get('x-forwarded-for') || req.ip,
            userAgent,
            path: req.nextUrl.pathname,
            timestamp: new Date().toISOString()
        });
    }
    
    // Optional: Redirect unauthenticated users (currently disabled)
    if (!req.auth && req.nextUrl.pathname !== "/") {
        // const newUrl = new URL("/", req.nextUrl.origin)
        // return Response.redirect(newUrl)
    }
    
    return response;
})

export const config = {
    matcher: ["/((?!api|_next/static|_next/image|favicon.ico).*)"],
}
